/* info_function.c
 *
 * Copyright (C) 2005, 2006, 2007 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file
   \brief Information functions.
   \author Stephane Germain <germste@gmail.com>
*/

#include "libirt.h"
#if HAVE_CONFIG_H
#  include <config.h>
#endif
#include <math.h>
#include <gsl/gsl_spline.h>

/**
   \brief Compute the items' information curves from the response functions.

   @param[in] probs A matrix (items x classes) of response functions.
   @param[in] quad_points A vector(classes) with the middle points of each class.
   @param[out] infos A matrix (items x classes) of items' information curves.
   @param[out] test_info A vector (classes) with test information curve.

   \warning The memory for the outputs must be allocated before.
*/
void
info_from_probs (gsl_matrix *probs, gsl_vector *quad_points, 
		 gsl_matrix *infos, gsl_vector *test_info)
{
  int nbr_item = probs->size1, nbr_quad = probs->size2, i, k;
  double prob, deriv, info;
  gsl_interp_accel *acc = gsl_interp_accel_alloc ();
  gsl_spline *spline = gsl_spline_alloc (gsl_interp_cspline, nbr_quad);

  for (i = 0; i < nbr_item; i++)
    {
      gsl_spline_init (spline, quad_points->data, probs->data+i*nbr_quad, nbr_quad);

      for (k = 0; k < nbr_quad; k++)
	{
	  prob = gsl_spline_eval(spline, gsl_vector_get(quad_points, k), acc);
	  deriv = gsl_spline_eval_deriv(spline, gsl_vector_get(quad_points, k), acc);
	  gsl_matrix_set(infos, i, k, deriv*deriv/(prob*(1-prob)));
	}
    }

  /* compute the test info */
  for (k = 0; k < nbr_quad; k++)
    {
      info = 0;
      for (i = 0; i < nbr_item; i++)
	{
	  info += gsl_matrix_get(infos, i, k);
	}
      gsl_vector_set(test_info, k, info);
    }

  gsl_spline_free (spline);
  gsl_interp_accel_free (acc);
}

/**
   \brief Compute the options' information curves from the response functions
   for a multivariate model.

   @param[in] probs A matrix (options x classes) of response functions.
   @param[in] quad_points A vector(classes) with the middle points of each class.
   @param[in] nbr_options A vector(items) with the number of option of each items.
   @param[in] items_pos A vector(items) with the position of the first option of each item.
   @param[out] options_infos A matrix (options x classes) of options' information curves.
   @param[out] infos A matrix (items x classes) of items' information curves.
   @param[out] test_info A vector (classes) with test information curve.

   \warning The memory for the outputs must be allocated before.
*/
void
info_from_probs_mc (gsl_matrix *probs, gsl_vector *quad_points,
		    gsl_vector_int *nbr_options, gsl_vector_int *items_pos,
		    gsl_matrix *options_infos, gsl_matrix *infos,
		    gsl_vector *test_info)
{
  int nbr_item = nbr_options->size, nbr_quad = quad_points->size, 
    nbr_option_tot = probs->size1, i, k, o, nbr_option, pos;
  double prob, deriv, info, item_info, tot_info;
  gsl_interp_accel *acc = gsl_interp_accel_alloc ();
  gsl_spline *spline = gsl_spline_alloc (gsl_interp_cspline, nbr_quad);

  /* compute the options' information curves */
  for (i = 0; i < nbr_option_tot; i++)
    {
      gsl_spline_init (spline, quad_points->data, probs->data+i*nbr_quad, nbr_quad);

      for (k = 0; k < nbr_quad; k++)
	{
	  prob = gsl_spline_eval(spline, gsl_vector_get(quad_points, k), acc);
	  deriv = gsl_spline_eval_deriv(spline, gsl_vector_get(quad_points, k), acc);
	  gsl_matrix_set(options_infos, i, k, deriv*deriv/(prob*prob));
	}
    }

  /* compute the items' information curves */
  for (k = 0; k < nbr_quad; k++)
    {  
      tot_info = 0;
      for (i = 0; i < nbr_item; i++)
	{
	  item_info = 0;
	  nbr_option = gsl_vector_int_get(nbr_options, i);
	  pos = gsl_vector_int_get(items_pos, i);
	  for (o = 0; o < nbr_option; o++)
	    {
	      prob = gsl_matrix_get(probs, pos+o, k);
	      info = gsl_matrix_get(options_infos, pos+o, k);
	      item_info += prob * info;
	    }
	  gsl_matrix_set(infos, i, k, item_info);
	  tot_info += item_info;
	}
      gsl_vector_set(test_info, k, tot_info);
    }

  gsl_spline_free (spline);
  gsl_interp_accel_free (acc);
}

