/* pmmle_wave.c
 *
 * Copyright (C) 2005 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file
   \brief Functions to estimate the item parameters by PMMLE
   (Penalized Maximal Marginal Likelihood).

   The functional estimations are done by a wavelet decomposition,
   and then by using a root finding algorithm on the wavelet coefficients.

   \author Stephane Germain <germste@gmail.com>
*/

#include "libirt.h"

#include <stdio.h>
#include <math.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_multiroots.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_wavelet.h>

/**
   \brief Used to passed extra parameter to \em mple_wave_fdfdf2.

   This is used to comply with the root finding functions in
   the gsl (GNU scientific library).
*/
typedef struct
{
  /** \brief The prior weights of each quadrature class. */
  gsl_vector *quad_weights;

  /** \brief The expected number of subject in each quadrature classes. */
  gsl_vector *quad_sizes;

  /** \brief The expected number of subject in each quadrature classes having a success. */
  gsl_vector *quad_freqs;

  /** \brief The penalizing factor */
  double smooth_factor;

  /** \brief Used to do the inverse wavelet transformation. */
  gsl_wavelet *wave;

  /** \brief Used to do the inverse wavelet transformation. */
  gsl_wavelet_workspace *work;

  /** \brief The wavelets evaluated at each middle point. */
  gsl_matrix *wavelets;

  /** \brief The (2)th derivative of the wavelets evaluated at each middle point. */
  gsl_matrix *deriv_wavelets;

  /** \brief A place to store the inverse wavelet transform
      (ie the logit of the response function). */
  gsl_vector *logit_space;

  /** \brief A place to store the (2)th derivative of the logit. */
  gsl_vector *deriv_logit_space;

} mple_wave_struct;

/**
   \brief Compute the log likelihood, gradient and Hessian of the wavelet coefficients.

   @param[in] par_wave The wavelet coefficients.
   @param[in] params The extra parameter to passes to the function.
   @param[out] f The log likelihood.
   @param[out] df The gradient of the log likelihood.
   @param[out] df2 The Hessian of the log likelihood.

   This function is not used directly by the root finding functions,
   but by others functions that comply with the gsl.

   \return GSL_SUCCESS for success.
*/
int
mple_wave_fdfdf2 (const gsl_vector * par_wave, void *params,
		  double *f, gsl_vector * df, gsl_matrix * df2)
{
  /* gsl_vector *quad_weights = ((mple_wave_struct *) params)->quad_weights; */
  gsl_vector *quad_sizes = ((mple_wave_struct *) params)->quad_sizes;
  gsl_vector *quad_freqs = ((mple_wave_struct *) params)->quad_freqs;
  double smooth_factor = ((mple_wave_struct *) params)->smooth_factor;
  gsl_wavelet *wave = ((mple_wave_struct *) params)->wave;
  gsl_wavelet_workspace *work = ((mple_wave_struct *) params)->work;
  gsl_matrix *wavelets = ((mple_wave_struct *) params)->wavelets;
  gsl_matrix *deriv_wavelets = ((mple_wave_struct *) params)->deriv_wavelets;
  gsl_vector *logit_space = ((mple_wave_struct *) params)->logit_space;
  gsl_vector *deriv_logit_space =
    ((mple_wave_struct *) params)->deriv_logit_space;
  double size, freq, logit, deriv_logit, prob, grad, hess, weight;
  int k, nbr_quad, s, t;

  nbr_quad = par_wave->size;

  /* copy the wavelet coefficients and do the inverse wavelet transform */
  for (k = 0; k < nbr_quad; k++)
    {
      gsl_vector_set (logit_space, k, gsl_vector_get (par_wave, k));
    }
  gsl_wavelet_transform_inverse (wave, logit_space->data, 1, nbr_quad, work);

  /* compute the (2)th derivative of the logit */
  for (k = 0; k < nbr_quad; k++)
    {
      deriv_logit = 0;
      for (s = 0; s < nbr_quad; s++)
	deriv_logit += gsl_vector_get (par_wave, s)
	  * gsl_matrix_get (deriv_wavelets, s, k);
      gsl_vector_set (deriv_logit_space, k, deriv_logit);
    }

  /* reset to zero */
  if (f)
    *f = 0;
  if (df)
    gsl_vector_set_all (df, 0.0);
  if (df2)
    gsl_matrix_set_all (df2, 0.0);

  /* sum over the classes */
  for (k = 0; k < nbr_quad; k++)
    {
      logit = gsl_vector_get (logit_space, k);
      prob = logistic (logit);
      size = gsl_vector_get (quad_sizes, k);
      weight = 1; /*gsl_vector_get (quad_weights, k);*/
      freq = gsl_vector_get (quad_freqs, k);
      deriv_logit = gsl_vector_get (deriv_logit_space, k);

      /* update the log likelihood */
      if (f) {
	*f += freq * log (prob) 
	  + (size - freq) * log (1 - prob)
	  - smooth_factor * weight * deriv_logit * deriv_logit;
	if(gsl_isnan((*f))) return GSL_FAILURE;
      }

      /* update the gradient */
      if (df)
	for (s = 0; s < nbr_quad; s++)
	  {
	    grad = gsl_vector_get (df, s);
	    grad += (freq - size * prob) * gsl_matrix_get (wavelets, s, k);
	    grad -= smooth_factor * weight * 2 * deriv_logit
	      * gsl_matrix_get (deriv_wavelets, s, k);
	    gsl_vector_set (df, s, grad);
	    if(gsl_isnan(grad)) return GSL_FAILURE;
	  }

      /* update the Hessian */
      if (df2)
	for (s = 0; s < nbr_quad; s++)
	  for (t = s; t < nbr_quad; t++)
	    {
	      hess = gsl_matrix_get (df2, s, t);
	      hess -= prob * (1 - prob) * size 
		* gsl_matrix_get (wavelets, s, k)
		* gsl_matrix_get (wavelets, t, k);
	      hess -= smooth_factor * weight * 2
		* gsl_matrix_get (deriv_wavelets, s, k)
		* gsl_matrix_get (deriv_wavelets, t, k);
	      gsl_matrix_set (df2, s, t, hess);
	      if(gsl_isnan(hess)) return GSL_FAILURE;
	    }
    }

  /* copy the lower half of the Hessian */
  if (df2)
    for (s = 0; s < nbr_quad; s++)
      for (t = 0; t < s; t++)
	gsl_matrix_set (df2, s, t, gsl_matrix_get (df2, t, s));

  return GSL_SUCCESS;
}

/**
   \brief Compute the log likelihood and gradient of the wavelet coefficients.

   @param[in] par_wave The wavelet coefficients.
   @param[in] params The extra parameter to passes to the function.
   @param[out] f The log likelihood.
   @param[out] df The gradient of the log likelihood.

   This function is just a wrapper around \em mple_wavefdfdf2
   to be used by the root finding functions in the gsl.
*/
int
mple_wave_fdf (const gsl_vector * par_wave, void *params,
	       double *f, gsl_vector * df)
{
  return mple_wave_fdfdf2 (par_wave, params, f, df, NULL);
}

/**
   \brief Compute the log likelihood gradient and Hessian of the wavelet coefficients.

   @param[in] par_wave The wavelet coefficients.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df The gradient of the log likelihood.
   @param[out] df2 The Hessian of the log likelihood.

   This function is just a wrapper around \em mple_wavefdfdf2
   to be used by the root finding functions in the gsl.

   \return GSL_SUCCESS for success.
*/
int
mple_wave_dfdf2 (const gsl_vector * par_wave, void *params,
		 gsl_vector * df, gsl_matrix * df2)
{
  return mple_wave_fdfdf2 (par_wave, params, NULL, df, df2);
}

/**
   \brief Compute the log likelihood of the wavelet coefficients.

   @param[in] par_wave The wavelet coefficients.
   @param[in] params The extra parameter to passes to the function.

   This function is just a wrapper around \em mple_wavefdfdf2
   to be used by the root finding functions in the gsl.

   \return The log likelihood.
*/
double
mple_wave_f (const gsl_vector * par_wave, void *params)
{
  double f;
  mple_wave_fdfdf2 (par_wave, params, &f, NULL, NULL);
  return f;
}

/**
   \brief Compute the log likelihood gradient of the wavelet coefficients.

   @param[in] par_wave The wavelet coefficients.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df The gradient of the log likelihood.

   This function is just a wrapper around \em mple_wavefdfdf2
   to be used by the root finding functions in the gsl.

   \return GSL_SUCCESS for success.
*/
int
mple_wave_df (const gsl_vector * par_wave, void *params, gsl_vector * df)
{
  return mple_wave_fdfdf2 (par_wave, params, NULL, df, NULL);
}

/**
   \brief Compute the log likelihood Hessian of the wavelet coefficients.

   @param[in] par_wave The wavelet coefficients.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df2 The Hessian of the log likelihood.

   This function is just a wrapper around \em mple_wavefdfdf2
   to be used by the root finding functions in the gsl.

   \return GSL_SUCCESS for success.
*/
int
mple_wave_df2 (const gsl_vector * par_wave, void *params, gsl_matrix * df2)
{
  return mple_wave_fdfdf2 (par_wave, params, NULL, NULL, df2);
}

/**
   \brief Does the maximization step of the EM algorithm to
   estimate the response functions by PMMLE (Penalized Maximum Marginal Likelihood).

   @param[in] max_iter The maximum number of Newton iterations performed for each item.
   @param[in] prec The desired precision of each wavelet parameter estimate.
   @param[in] smooth_factor The factor to the penality term.
   @param[in] quad_freqs A matrix(items x classes) with the expected number of
   subjects in the class having a success at the item.
   @param[in] quad_sizes A vector (classes) with the expected number of
   subjects in the class.
   @param[in] params The extra parameter to passes to the function.
   @param[in,out] probs A matrix(items x classes) with the estimated response functions.
   They should be initialize first.
   @param[out] probs_stddev matrix(items x classes) with the standard error
   of the logit response functions.
   @param[in] ignore A vector(items) with a flag of 1 if the item should be ignore.
   @param[out] notconverge A vector(items) of flag set for the items that didn't converged.
   @param[out] tot_pml The vaulue of the PML.

   \return The number of item that did not converged.
   
   \warning The memory for the response functions should be allocated before.
*/
int
mple_wave (int max_iter, double prec, double smooth_factor,
	   gsl_matrix * quad_freqs, gsl_vector * quad_sizes, 
	   mple_wave_struct * params,
	   gsl_matrix * probs, gsl_matrix * probs_stddev,
	   gsl_vector_int * ignore, gsl_vector_int * notconverge,
	   double * tot_pml)
{
  const gsl_multiroot_fdfsolver_type *algo;
  gsl_multiroot_fdfsolver *solver;
  int status, iter, nbr_item, i, k, s, t, ret_val, nbr_quad = quad_freqs->size2;
  gsl_multiroot_function_fdf FDF;
  gsl_vector *par_wave = gsl_vector_alloc (nbr_quad);
  gsl_vector_view quad_freqs_i;

  /* used to compute the standard errors */
  gsl_matrix *df2 = gsl_matrix_alloc (nbr_quad, nbr_quad);
  gsl_permutation *lu_perm = gsl_permutation_alloc (nbr_quad);
  gsl_matrix *inv_df2 = gsl_matrix_alloc (nbr_quad, nbr_quad);
  int lu_sign;
  double var, sum_grad, prob;

  nbr_item = quad_freqs->size1;

  /* initalize the parameter and function to solve */
  params->smooth_factor = smooth_factor;
  FDF.f = &mple_wave_df;
  FDF.df = &mple_wave_df2;
  FDF.fdf = &mple_wave_dfdf2;
  FDF.n = nbr_quad;
  FDF.params = params;

  /* select the algorithm to used */
  algo = gsl_multiroot_fdfsolver_gnewton;
  /* allocate the solver */
  solver = gsl_multiroot_fdfsolver_alloc (algo, nbr_quad);

  ret_val = 0;

  *tot_pml = 0;

  /* for each item */
  for (i = 0; i < nbr_item; i++)
    {
      gsl_vector_int_set(notconverge, i, 0);

      /* ignore the degenerate items */
      if (ignore && gsl_vector_int_get(ignore, i)) continue;

      if (libirt_verbose > 3)
	printf ("item %d", i + 1);

      /* get the corresponding rows of freqs, sizes, probs and probs_stddev */
      quad_freqs_i = gsl_matrix_row (quad_freqs, i);
      params->quad_freqs = &quad_freqs_i.vector;
      params->quad_sizes = quad_sizes;

      /* set the starting values ... */
      /* ... take the logistic transformation ... */
      for (k = 0; k < nbr_quad; k++)
	{
	  gsl_vector_set (par_wave, k, logit (gsl_matrix_get (probs, i, k)));
	}
      /* ... and use a wavelet transform */
      gsl_wavelet_transform_forward (params->wave, par_wave->data, 1,
				     nbr_quad, params->work);

      /* set the solver */
      gsl_multiroot_fdfsolver_set (solver, &FDF, par_wave);

      /* iterate the solver */
      iter = 0;
      do
	{
	  iter++;

	  /* check if the hessian is singular */
	  status = 0;
	  for (k = 0; k < nbr_quad; k++)
	    {
	      if(0 == gsl_matrix_get (solver->J, k, k))
		{
		  status = GSL_EBADFUNC;
		  break;
		}
	    }
	  if (status) break;

	  status = gsl_multiroot_fdfsolver_iterate (solver);

	  if (libirt_verbose > 9)
	    {
	      sum_grad = 0;
	      for (k = 0; k < nbr_quad; k++) sum_grad += fabs(gsl_vector_get(solver->f, k));
	      printf ("\n At N-R iteration %d PML is %9.2e and sum|grad(PML)| is %8.2e.\n",
		      iter, mple_wave_f(par_wave, params),
		      sum_grad);
	    }

	  if (status)
	    break;

	  /* test for convergence */
	  status = gsl_multiroot_test_delta (solver->dx, solver->x, prec, 0);
	  /* status = gsl_multiroot_test_residual (solver->f, prec); */

	}
      while (status == GSL_CONTINUE && iter < max_iter);

      /* check if this item converged */
      if (status != GSL_SUCCESS)
	{
	  ret_val++;
	  gsl_vector_int_set(notconverge, i, 1);
	}

      if (libirt_verbose > 3)
	{
	  if (status == GSL_CONTINUE)
	    printf (" did not converged (max iter)");
	  else if (status == GSL_EBADFUNC)
	    printf (" did not converged (singular point)");
	  else if (status == GSL_ENOPROG)
	    printf (" did not converged (no progress)");
	  else if (status == GSL_SUCCESS)
	    printf (" converged (success)");
	  else
	    printf (" unknow status");
	  printf (" after %d iterations.\n", iter);
	  fflush (stdout);
	}

      /* copy the solution found */
      /* ... use an inverse  wavelet transform and ... */
      gsl_wavelet_transform_inverse (params->wave, solver->x->data, 1,
				     nbr_quad, params->work);
      for (k = 0; k < nbr_quad; k++)
	{
	  /* ... take the logistic transformation */
	  prob = logistic (gsl_vector_get (solver->x, k));
	  
	  /* reset the probabilities inside the open interval (0,1)
	     because of rounding errors 0 and 1 could have been attain 
	     in the previous iteration */
	  if (prob < VERY_SMALL_PROB) prob = VERY_SMALL_PROB;
	  if (prob > 1 - VERY_SMALL_PROB) prob = 1 - VERY_SMALL_PROB;
	  
	  gsl_matrix_set (probs, i, k, prob);
	}

      /* compute the standard errors */
      if (probs_stddev)
	{
	  /* get the Hessian */
	  mple_wave_df2 (solver->x, params, df2);

	  /* inverse it */
	  gsl_linalg_LU_decomp (df2, lu_perm, &lu_sign);
	  gsl_linalg_LU_invert (df2, lu_perm, inv_df2);

	  /* inverse wavelet transform the standard errors (var(sum) = sum(cov)) */
	  for (k = 0; k < nbr_quad; k++)
	    {
	      var = 0;
	      for (s = 0; s < nbr_quad; s++)
		for (t = 0; t < nbr_quad; t++)
		  var -= gsl_matrix_get (params->wavelets, s, k)
		    * gsl_matrix_get (params->wavelets, t, k)
		    * gsl_matrix_get (inv_df2, s, t);

	      /* compute the standard error of the logistic response function */
	      var = sqrt (var);

	      /* compute an upper bound for the standard error of the response function */
	      var = logistic (fabs (gsl_vector_get (solver->x, k)))
		- logistic (fabs (gsl_vector_get (solver->x, k)) - var);

	      gsl_matrix_set (probs_stddev, i, k, var);
	    }
	}

      /* compute the total criteria */
      *tot_pml += mple_wave_f(par_wave, params);
    }

  /* free the memory */
  gsl_multiroot_fdfsolver_free (solver);
  gsl_vector_free (par_wave);
  gsl_matrix_free (df2);
  gsl_permutation_free (lu_perm);
  gsl_matrix_free (inv_df2);

  return ret_val;
}

/**
   \brief Estimate the response functions by PMMLE (Penalized Maximum Marginal Likelihood).

   @param[in] max_em_iter The maximum number of EM iterations. At least 20 iteration are made.
   @param[in] max_nr_iter The maximum number of Newton iterations performed
   for each item at each EM iteration.
   @param[in] prec The relative change in the likelihood to stop the EM algorithm.
   This value divided by 10 is also the desired precision of each parameter estimate.
   @param[in] smooth_factor The factor to the penality term.
   @param[in] patterns A matrix(patterns x items) of binary responses.
   @param[in] counts A vector(patterns) with the count of each pattern.
   If NULL the counts are assumed to be all 1.
   @param[in] quad_points A vector(classes) with the middle points of each quadrature class.
   @param[in] quad_weights A vector(classes) with the prior weights of each quadrature class.
   @param[in,out] probs A matrix(items x classes) with the estimated response functions.
   They should be initialize first.
   @param[out] probs_stddev matrix(items x classes) with the standard error
   of the logit response functions.
   @param[in] ignore A vector(items) of ignore flag.
   @param[out] nbr_notconverge The number of items that didn't converged.
   @param[out] notconverge A vector(items) of flag set for the items that didn't converged.
   @param[in] adjust_weights Controls whether adjust the quadrature weights after each iteration.

   \return The number of item that did not converged at the last EM iteration.
   
   \warning The memory for the response functions should be allocated before.
*/
int
em_mple_wave (int max_em_iter, int max_nr_iter, double prec,
	      double smooth_factor, gsl_matrix_int * patterns,
	      gsl_vector * counts, gsl_vector * quad_points,
	      gsl_vector * quad_weights, gsl_matrix * probs,
	      gsl_matrix * probs_stddev, gsl_vector_int * ignore, 
	      int * nbr_notconverge, gsl_vector_int * notconverge,
	      int adjust_weights)
{
  int em_iter, nbr_quad, nbr_pattern, nbr_item, ret_val, s, k, j, i;
  double deriv, step, mllk, mllk_old=0, prob, nbr_subject;
  gsl_matrix *quad_freqs;
  gsl_vector *quad_sizes;
  gsl_matrix *post;
  mple_wave_struct params;

  nbr_quad = quad_points->size;
  nbr_pattern = patterns->size1;
  nbr_item = patterns->size2;

  nbr_subject = 0;
  /* count the number of subject */
  for(j = 0; j < nbr_pattern; j++)
    nbr_subject += counts ? gsl_vector_get(counts, j) : 1;

  /* adjust the smoothing factor for the quadratures widths factor */
  smooth_factor *= (gsl_vector_get(quad_points, nbr_quad - 1) 
                    - gsl_vector_get(quad_points, 0)) / (nbr_quad - 1);

  /* allocate the memory */
  quad_freqs = gsl_matrix_alloc (nbr_item, nbr_quad);
  quad_sizes = gsl_vector_alloc (nbr_quad);
  post = gsl_matrix_alloc (nbr_pattern, nbr_quad);
  params.logit_space = gsl_vector_alloc (nbr_quad);
  params.deriv_logit_space = gsl_vector_alloc (nbr_quad);

  params.quad_weights = quad_weights;

  /* initialize the wavelets transform */
  params.wave = gsl_wavelet_alloc (gsl_wavelet_daubechies, 10);
  params.work = gsl_wavelet_workspace_alloc (nbr_quad);

  /* compute the wavelets and theirs (2)th derivatives */
  /* allocate the memory */
  params.wavelets = gsl_matrix_alloc (nbr_quad, nbr_quad);
  params.deriv_wavelets = gsl_matrix_alloc (nbr_quad, nbr_quad);
  /* set to zero */
  gsl_matrix_set_all (params.wavelets, 0);
  for (s = 0; s < nbr_quad; s++)
    {
      /* select the wavelet */
      gsl_matrix_set (params.wavelets, s, s, 1);
      /* compute the wavelet */
      gsl_wavelet_transform_inverse (params.wave,
				     params.wavelets->data + s * nbr_quad,
				     1, nbr_quad, params.work);
      /* compute its (2)th derivatives */
      /** \todo Compute more accurates wavelets derivatives */
      for (k = 1; k < nbr_quad - 1; k++)
	{
	  deriv = gsl_matrix_get (params.wavelets, s, k + 1)
	    - 2 * gsl_matrix_get (params.wavelets, s, k)
	    + gsl_matrix_get (params.wavelets, s, k - 1);
	  step = (gsl_vector_get (quad_points, k + 1) -
		  gsl_vector_get (quad_points, k - 1)) / 2;
	  step *= step;
	  deriv /= step;
	  gsl_matrix_set (params.deriv_wavelets, s, k, deriv);
	}
      gsl_matrix_set (params.deriv_wavelets, s, k, deriv);
      gsl_matrix_set (params.deriv_wavelets, s, 0,
		      gsl_matrix_get (params.deriv_wavelets, s, 1));
    }

  /* reset the probabilities inside the open interval (0,1) */
  for (i = 0; i < nbr_item; i++)
    for (k = 0; k < nbr_quad; k++)
      {
	prob = gsl_matrix_get(probs, i, k);
	if (prob < VERY_SMALL_PROB) prob = VERY_SMALL_PROB;
	if (prob > 1 - VERY_SMALL_PROB) prob = 1 - VERY_SMALL_PROB;
        gsl_matrix_set(probs, i, k, prob);
      }

  ret_val = 0;

  /* EM iterations */

  for (em_iter = 1; em_iter <= max_em_iter; em_iter++)
    {
      /* E (estimation) step */

      if (libirt_verbose > 2)
	printf ("\nEM iteration %d\n", em_iter);

      /* compute the posterior prob */
      posteriors (patterns, probs, quad_weights, post);

      /* compute the expected sizes and frequencies */
      frequencies (patterns, counts, post, probs, quad_sizes, quad_freqs);

      /* print debugging information */
      if (libirt_verbose > 5)
	{
	  for (i = 0; i < nbr_item; i++)
	    {
	      printf("Probabilities for item %d :\n", i+1);
	      for (k = 0; k < nbr_quad; k++)
		printf(" %8.2e", gsl_matrix_get(probs,i,k));
	      printf("\n");
	    }
	  for (j = 0; j < nbr_pattern; j++)
	    {
	      printf("Posterior for pattern %d :\n", j+1);
	      for (k = 0; k < nbr_quad; k++)
		printf(" %8.2e", gsl_matrix_get(post,j,k));
	      printf("\n");
	    }
	  printf("Sizes :\n");
	  for (k = 0; k < nbr_quad; k++)
	    printf(" %8.2e", gsl_vector_get(quad_sizes,k));
	  printf("\n");
	  for (i = 0; i < nbr_item; i++)
	    {
	      printf("Frequencies for item %d :\n", i+1);
	      for (k = 0; k < nbr_quad; k++)
		printf(" %8.2e", gsl_matrix_get(quad_freqs,i,k));
	      printf("\n");
	    }
	}

      /* M (maximisation) step */
      *nbr_notconverge = mple_wave (max_nr_iter, prec/10, smooth_factor,
				    quad_freqs, quad_sizes,
				    &params, probs, probs_stddev, ignore,
				    notconverge, &mllk);

      if(gsl_isnan(mllk)) {
	if (libirt_verbose > 1) printf("NAN error ! Stopping.\n");
	break;
      }

      if(adjust_weights)
	adjust_quad_weights (nbr_subject, quad_sizes, quad_points, quad_weights);

      if (libirt_verbose > 2)
	printf("MLLK = %10.3e %%CHANGE = %9.3e\n", mllk, fabs((mllk-mllk_old)/mllk));

      /* if the change in the maximum log likelihood is small then exit */
      if (fabs((mllk-mllk_old)/mllk) < prec && em_iter >= 20) break;

      mllk_old = mllk;
    }

  /* check if the EM algo converged */
  if (em_iter <= max_em_iter && !gsl_isnan(mllk)) ret_val = 1;
  else ret_val = 0;

  if (libirt_verbose > 0 && ret_val == 0)
    printf("The EM algorithm didn't converged after %d iterations.\n", em_iter-1);

  if (libirt_verbose > 0 && ret_val == 1)
    printf("The EM algorithm converged after %d iterations.\n", em_iter);

  /* free the memory */
  gsl_matrix_free (quad_freqs);
  gsl_vector_free (quad_sizes);
  gsl_matrix_free (post);
  gsl_wavelet_free (params.wave);
  gsl_wavelet_workspace_free (params.work);
  gsl_matrix_free (params.wavelets);
  gsl_matrix_free (params.deriv_wavelets);
  gsl_vector_free (params.logit_space);
  gsl_vector_free (params.deriv_logit_space);

  return ret_val;
}
