/* mle_abilitt_3plm.c
 *
 * Copyright (C) 2005 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file
   \brief Functions to estimate the abilities by ML.
   \author Stephane Germain <germste@gmail.com>
*/

#include "libirt.h"

#include <stdio.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_roots.h>

/**
   \brief Used to passed extra parameter to \em mle_ability_3plm_fdfdf2.

   This is used to comply with the root finding functions in
   the gsl (GNU scientific library).
*/
typedef struct
{
  /** \brief The pattern of the subject. */
  gsl_vector_int *pattern;

  /** \brief The slope parameters. */
  gsl_vector *slopes;

  /** \brief The threshold parameters. */
  gsl_vector *thresholds;

  /** \brief The asymptote parameters. */
  gsl_vector *asymptotes;

} mle_ability_3plm_struct;

/**
   \brief Compute the log likelihood, gradient and Hessian of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.
   @param[out] f The log likelihood.
   @param[out] df The gradient of the log likelihood.
   @param[out] df2 The Hessian of the log likelihood.

   This function is not used directly by the root finding functions,
   but by others functions that comply with the gsl.
*/
void
mle_ability_3plm_fdfdf2 (double ability, void *params,
			 double *f, double *df, double *df2)
{
  gsl_vector_int *pattern = ((mle_ability_3plm_struct *) params)->pattern;
  gsl_vector *slopes = ((mle_ability_3plm_struct *) params)->slopes;
  gsl_vector *thresholds = ((mle_ability_3plm_struct *) params)->thresholds;
  gsl_vector *asymptotes = ((mle_ability_3plm_struct *) params)->asymptotes;
  int i, nbr_item, resp;
  double slope, asymptote, prob, prob2, L, L1, L11;

  nbr_item = pattern->size;
  L = 0;
  L1 = 0;
  L11 = 0;
  /* for each item */
  for (i = 0; i < nbr_item; i++)
    {
      resp = gsl_vector_int_get (pattern, i);
      slope = gsl_vector_get (slopes, i);
      asymptote = asymptotes ? gsl_vector_get (asymptotes, i) : 0;
      prob =
	prob_3plm (ability, slope, gsl_vector_get (thresholds, i),
		   asymptote);
      prob2 =
	prob_3plm (ability, slope, gsl_vector_get (thresholds, i), 0);
      if (resp != BLANK)
	{
	  /* update the likelihood */
	  L += resp ? log (prob) : log (1 - prob);
	  /* update the gradient */
	  L1 += slope * (resp - prob) * prob2 / prob;
	  /* update the Hessian */
	  L11 -= slope * slope * (1 - prob) * prob2 * prob2 / prob;
	}
      else
	{
	  /* adjust the likelihood, gradient and Hessian 
	      for the missing data (P^2 + (1-P)^2). 
	      got tired for the Hessian and used maxima ! */
	  L += log (prob * prob + (1 - prob) * (1 - prob));
	  L1 += 2 * slope * (1 - prob) * prob2 * (2 * prob - 1)
	    / (prob * prob + (1 - prob) * (1 - prob));
	  L11 += 2 * slope * slope * prob2
	    * (1+asymptote+prob*(-6+prob*(10-2*asymptote+prob*(-8+prob*4))))
	    / ((1-asymptote)*(2*prob*prob-2*prob+1));
	}
    }

  if (f)
    *f = L;
  if (df)
    *df = L1;
  if (df2)
    *df2 = L11;
}

/**
   \brief Compute the log likelihood of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.

   This function is just a wrapper around \em mle_ability_3plm_fdfdf2
   to be used by the root finding functions in the gsl.

   \return The log likelihood.
*/
double
mle_ability_3plm_f (double ability, void *params)
{
  double f;
  mle_ability_3plm_fdfdf2 (ability, params, &f, NULL, NULL);
  return f;
}

/**
   \brief Compute the log likelihood gradient of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.

   This function is just a wrapper around \em mle_ability_3plm_fdfdf2
   to be used by the root finding functions in the gsl.

   \return The log likelihood gradient.
*/
double
mle_ability_3plm_df (double ability, void *params)
{
  double df;
  mle_ability_3plm_fdfdf2 (ability, params, NULL, &df, NULL);
  return df;
}

/**
   \brief Compute the log likelihood Hessian of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.

   This function is just a wrapper around \em mle_ability_3plm_fdfdf2
   to be used by the root finding functions in the gsl.

   \return The log likelihood Hessian.
*/
double
mle_ability_3plm_df2 (double ability, void *params)
{
  double df2;
  mle_ability_3plm_fdfdf2 (ability, params, NULL, NULL, &df2);
  return df2;
}

/**
   \brief Compute the log likelihood gradient and Hessian of the pattern.

   @param[in] ability The ability level.
   @param[in] params The extra parameter to passes to the function.
   @param[out] df The gradient of the log likelihood.
   @param[out] df2 The Hessian of the log likelihood.

   This function is just a wrapper around \em mle_ability_3plm_fdfdf2
   to be used by the root finding functions in the gsl.
*/
void
mle_ability_3plm_dfdf2 (double ability, void *params, double *df, double *df2)
{
  mle_ability_3plm_fdfdf2 (ability, params, NULL, df, df2);
}

/**
   \brief Estimate the abilities by maximum likelihood.

   @param[in] max_iter The maximum number of Newton iterations performed for each pattern.
   @param[in] prec The desired precision of each parameter estimate.
   @param[in] patterns A matrix(patterns x items) of binary responses.
   @param[in] slopes A vector(items) with the slope parameters of each item.
   @param[in] thresholds A vector(items) with the threshold parameters of each item.
   @param[in] asymptotes A vector(items) with the asymptote parameters of each item.
   Can be NULL for the 2PLM.
   @param[in,out] abilities A vector(patterns) with the estimated abilities.
   They should be initialize first.
   @param[out] abilities_stddev A vector(items) with the standard errors of
   the estimated abilities.

   \return The number of pattern that did not converged.
*/
int
mle_abilities_3plm (int max_iter, double prec,
		    gsl_matrix_int * patterns,
		    gsl_vector * slopes,
		    gsl_vector * thresholds,
		    gsl_vector * asymptotes,
		    gsl_vector * abilities, gsl_vector * abilities_stddev)
{
  int nbr_pattern, nbr_item, j, status, iter, ret_val, i, degenerate;
  double ability, ability0, df2;
  gsl_vector_int_view pattern_j;
  const gsl_root_fdfsolver_type *T;
  gsl_root_fdfsolver *s;
  gsl_function_fdf FDF;
  mle_ability_3plm_struct params;

  /* set parameters and function to solve */
  params.slopes = slopes;
  params.thresholds = thresholds;
  params.asymptotes = asymptotes;
  FDF.fdf = &mle_ability_3plm_dfdf2;
  FDF.f = &mle_ability_3plm_df;
  FDF.df = &mle_ability_3plm_df2;
  FDF.params = &params;

  /* select the algorithm to used */
  T = gsl_root_fdfsolver_steffenson;
  /* allocate the solver */
  s = gsl_root_fdfsolver_alloc (T);

  nbr_pattern = patterns->size1;
  nbr_item = patterns->size2;

  ret_val = 0;

  /* for each pattern */
  for (j = 0; j < nbr_pattern; j++)
    {
      if (libirt_verbose > 3)
	printf ("pattern %d ", j + 1);

      /** detect the degenerates patterns. */
      degenerate = 1;
      for (i = 1; i < nbr_item; i++)
	if (gsl_matrix_int_get (patterns, j, i) !=
	    gsl_matrix_int_get (patterns, j, i - 1))
	  {
	    degenerate = 0;
	    break;
	  }

      /* set the starting value */
      ability = gsl_vector_get (abilities, j);

      /* get the corresponding row of patterns */
      pattern_j = gsl_matrix_int_row (patterns, j);
      params.pattern = &pattern_j.vector;

      /* set the solver */
      gsl_root_fdfsolver_set (s, &FDF, ability);

      /* iterate the solver */
      iter = 0;
      do
	{
	  iter++;
	  status = gsl_root_fdfsolver_iterate (s);
	  ability0 = ability;
	  ability = gsl_root_fdfsolver_root (s);

	  /* if the step is to big shrink it */
	  if (fabs (ability - ability0) > 0.5)
	    {
	      ability = ability0 + ((ability > ability0) ? 0.5 : -0.5);
	      gsl_root_fdfsolver_set (s, &FDF, ability);
	    }

	  /* test for convergence */
	  status = gsl_root_test_delta (ability, ability0, prec, 0);
	}
      while (status == GSL_CONTINUE && iter < max_iter);

      /* check if this pattern converged */
      if (status != GSL_SUCCESS)
	ret_val++;

      if (libirt_verbose > 3)
	{
	  if (status == GSL_CONTINUE)
	    {
	      if (degenerate)
		printf ("did not converged (degenerate)");
	      else
		printf ("did not converged (max iter)");
	    }
	  else if (status == GSL_EBADFUNC)
	    printf ("did not converged (singular point)");
	  else if (status == GSL_ENOPROG)
	    printf ("did not converged (no progress)");
	  else if (status == GSL_SUCCESS)
	    printf ("converged (success)");
	  else
	    printf ("unknow status");
	  printf (" after %d iterations.\n", iter);
	  fflush (stdout);
	}

      /* copy the solution found */
      gsl_vector_set (abilities, j, ability);

      /* copy the standard error */
      if (abilities_stddev)
	{
	  mle_ability_3plm_dfdf2 (ability, &params, NULL, &df2);
	  gsl_vector_set (abilities_stddev, j, sqrt (-1 / df2));
	}
    }

  /* free the memory */
  gsl_root_fdfsolver_free (s);

  return ret_val;
}
